#!/bin/bash

# disketteformat.sh
#
# Usage: disketteformat.sh <device file> <filesystem> <log file> <label>
#
# device file: The device file for the diskette, usually given as /dev/fd0
# filesystem: The integer representing the filesystem type, either 1 for vfat, or 4 for ext2 
# log file: The log file for recording the output of the diskette format script
# label: The optionally passed label string used to format the diskette with a label
#
# Return Codes:
# 1 - Error formating the diskette media
#
# Module History
#  00  11/21/2003  P. Provost - Changed to use /sbin/mkfs because /sbin is not in the root path
#  01  10/11/2004  P. Provost - Changed to use /sbin/mkdosfs -F 32 for FAT32 filesystems instead of /sbin/mkfs.msdos
#  02  12/15/2004  P. Provost - Put double quotes around $LABEL in case the given label contains spaces
#  03  01/10/2005  P. Provost - Added -I option to mkdosfs command to vfat format USB devices
#  04  01/10/2005  P. Provost - Added -F option to mkfs command to force ext2 format of USB devices 

DEVICEFILE=$1
FILESYSTEM=$2
LOGFILE=$3
LABEL=$4

echo "-> disketteformat.sh" >> $LOGFILE
echo "Formatting diskette at $DEVICEFILE for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> disketteformat.sh"
actzTrace "XMDA   F: disketteformat.sh: Formatting diskette at $DEVICEFILE for `date`" 

# Determine what filesystem is to be used to format the diskette
case "$FILESYSTEM" in
    # If the diskette is to be formatted with the dos filesystem 
    1)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then
            # Format a vfat diskette with no label
            echo "Formatting a vfat diskette with no label." >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting a vfat diskette with no label."
            
            if ! /sbin/mkdosfs -F 32 -v -I $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format a vfat diskette...exiting." >> $LOGFILE
                actzTrace "XMDA   F: disketteformat.sh: Couldn't format a vfat diskette...exiting." 
                
                exit 1
            fi
        else
            # Format a vfat diskette with a label
            echo "Formatting a vfat diskette with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting a vfat diskette with the label: $LABEL"
            
            if ! /sbin/mkdosfs -F 32 -v -I -n "$LABEL" $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format a vfat diskette with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: disketteformat.sh: Couldn't format a vfat diskette with a label...exiting."
                
                exit 1
            fi
        fi
    ;;
    # If the diskette is to be formatted with the ext2 filesystem
    4)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then
            # Format an ext2 diskette with no label
            echo "Formatting an ext2 diskette with no label." >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting an ext2 diskette with no label."
            
            if ! /sbin/mkfs -t ext2 -v -F $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format an ext2 diskette...exiting." >> $LOGFILE
                actzTrace "XMDA   F: disketteformat.sh: Couldn't format an ext2 diskette...exiting."
                
                exit 1
            fi
        else
            # Format an ext2 diskette with a label
            echo "Formatting an ext2 diskette with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: disketteformat.sh: Formatting an ext2 diskette with the label: $LABEL" 
            
            if ! /sbin/mkfs -t ext2 -v -F -L "$LABEL" $DEVICEFILE >> $LOGFILE 2>&1; then
                echo "Couldn't format an ext2 diskette with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: disketteformat.sh: Couldn't format an ext2 diskette with a label...exiting."
                
                exit 1
            fi
        fi
    ;;
esac

actzTrace "XMDA   T: <- disketteformat.sh"

echo "<- disketteformat.sh" >> $LOGFILE
echo "" >> $LOGFILE

exit 0

